// MouseCard.cpp: implementation of the CMouseCard class.
//
//////////////////////////////////////////////////////////////////////

#include "arch/frame/stdafx.h"
#include "arch/frame/aipc.h"
#include "arch/directx/dimouse.h"
#include "appleclock.h"

#include "mousecard.h"

extern CDIMouse g_cDIMouse;
extern CAppleClock *g_pBoard;

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

// Sets mouse mode
#define MOUSE_SET		0x00
// Reads mouse position
#define MOUSE_READ		0x10
// Services mouse interrupt
#define MOUSE_SERV		0x20
// Clears mouse positions to 0 (for delta mode)
#define MOUSE_CLEAR		0x30
// Sets mouse position to a user-defined pos
#define MOUSE_POS		0x40
// Resets mouse clamps to default values
// Sets mouse position to 0,0
#define MOUSE_INIT		0x50
// Sets mouse bounds in a window
#define MOUSE_CLAMP		0x60
// Sets mouse to upper-left corner of clamp win
#define MOUSE_HOME		0x70

// Set VBL Timing : 0x90 is 60Hz, 0x91 is 50Hz
#define MOUSE_TIME		0x90

#define BIT0		0x01
#define BIT1		0x02
#define BIT2		0x04
#define BIT3		0x08
#define BIT4		0x10
#define BIT5		0x20
#define BIT6		0x40
#define BIT7		0x80

IMPLEMENT_DYNAMIC( CMouseCard, CCard );

const static BYTE rom[] =
{
	0x2C, 0x58, 0xFF, 0x70, 0x1B, 0x38, 0x90, 0x18,	// 0x0000
	0xB8, 0x50, 0x15, 0x01, 0x20, 0xF4, 0xF4, 0xF4,
	0xF4, 0x00, 0xB3, 0xC4, 0x9B, 0xA4, 0xC0, 0x8A,
	0xDD, 0xBC, 0x48, 0xF0, 0x53, 0xE1, 0xE6, 0xEC,
	0x08, 0x78, 0x8D, 0xF8, 0x07, 0x48, 0x98, 0x48,
	0x8A, 0x48, 0x20, 0x58, 0xFF, 0xBA, 0xBD, 0x00,
	0x01, 0xAA, 0x08, 0x0A, 0x0A, 0x0A, 0x0A, 0x28,
	0xA8, 0xAD, 0xF8, 0x07, 0x8E, 0xF8, 0x07, 0x48,
	0xA9, 0x08, 0x70, 0x67, 0x90, 0x4D, 0xB0, 0x55,
	0x29, 0x01, 0x09, 0xF0, 0x9D, 0x38, 0x06, 0xA9,
	0x02, 0xD0, 0x40, 0x29, 0x0F, 0x09, 0x90, 0xD0,
	0x35, 0xFF, 0xFF, 0xB9, 0x83, 0xC0, 0x29, 0xFB,
	0x99, 0x83, 0xC0, 0xA9, 0x3E, 0x99, 0x82, 0xC0,
	0xB9, 0x83, 0xC0, 0x09, 0x04, 0x99, 0x83, 0xC0,
	0xB9, 0x82, 0xC0, 0x29, 0xC1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0xF0, 0x0A, 0x6A, 0x90,
	0x75, 0x68, 0xAA, 0x68, 0xA8, 0x68, 0x28, 0x60,	// 0x0080
	0x18, 0x60, 0x29, 0x01, 0x09, 0x60, 0x9D, 0x38,
	0x06, 0xA9, 0x0E, 0x9D, 0xB8, 0x05, 0xA9, 0x01,
	0x48, 0xD0, 0xC0, 0xA9, 0x0C, 0x9D, 0xB8, 0x05,
	0xA9, 0x02, 0xD0, 0xF4, 0xA9, 0x30, 0x9D, 0x38,
	0x06, 0xA9, 0x06, 0x9D, 0xB8, 0x05, 0xA9, 0x00,
	0x48, 0xF0, 0xA8, 0xC9, 0x10, 0xB0, 0xD2, 0x9D,
	0x38, 0x07, 0x90, 0xEA, 0xA9, 0x04, 0xD0, 0xEB,
	0xA9, 0x40, 0xD0, 0xCA, 0xA4, 0x06, 0xA9, 0x60,
	0x85, 0x06, 0x20, 0x06, 0x00, 0x84, 0x06, 0xBA,
	0xBD, 0x00, 0x01, 0xAA, 0x0A, 0x0A, 0x0A, 0x0A,
	0xA8, 0xA9, 0x20, 0xD0, 0xC9, 0xA9, 0x70, 0xD0,
	0xC5, 0x48, 0xA9, 0xA0, 0xD0, 0xA8, 0x29, 0x0F,
	0x09, 0xB0, 0xD0, 0xBA, 0xA9, 0xC0, 0xD0, 0xB6,
	0xA9, 0x02, 0xD0, 0xB7, 0xA2, 0x03, 0x38, 0x60,
	0xFF, 0xFF, 0xFF, 0xD6, 0xFF, 0xFF, 0xFF, 0x01,
	0x98, 0x48, 0xA5, 0x06, 0x48, 0xA5, 0x07, 0x48,	// 0x0100
	0x86, 0x07, 0xA9, 0x27, 0x85, 0x06, 0x20, 0x58,
	0xFC, 0xA0, 0x00, 0xB1, 0x06, 0xF0, 0x06, 0x20,
	0xED, 0xFD, 0xC8, 0xD0, 0xF6, 0x68, 0x85, 0x07,
	0x68, 0x85, 0x06, 0x68, 0xA8, 0xD0, 0x5B, 0xC1,
	0xF0, 0xF0, 0xEC, 0xE5, 0xCD, 0xEF, 0xF5, 0xF3,
	0xE5, 0x8D, 0xC3, 0xEF, 0xF0, 0xF9, 0xF2, 0xE9,
	0xE7, 0xE8, 0xF4, 0xA0, 0xB1, 0xB9, 0xB8, 0xB3,
	0xA0, 0xE2, 0xF9, 0xA0, 0xC1, 0xF0, 0xF0, 0xEC,
	0xE5, 0xA0, 0xC3, 0xEF, 0xED, 0xF0, 0xF5, 0xF4,
	0xE5, 0xF2, 0xAC, 0xA0, 0xC9, 0xEE, 0xE3, 0xAE,
	0x8D, 0x8D, 0xC2, 0xE1, 0xE3, 0xE8, 0xED, 0xE1,
	0xEE, 0xAF, 0xCD, 0xE1, 0xF2, 0xEB, 0xF3, 0xAF,
	0xCD, 0xE1, 0xE3, 0xCB, 0xE1, 0xF9, 0x8D, 0x00,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0x30, 0x0C, 0xF0, 0x80,
	0xD0, 0x09, 0xA9, 0x00, 0x9D, 0xB8, 0x05, 0x48,	// 0x0180
	0xF0, 0xE6, 0x60, 0xBD, 0x38, 0x07, 0x29, 0x0F,
	0x09, 0x20, 0x9D, 0x38, 0x07, 0x8A, 0x48, 0x48,
	0x48, 0x48, 0xA9, 0xAA, 0x48, 0xBD, 0x38, 0x06,
	0x48, 0xA9, 0x0C, 0x9D, 0xB8, 0x05, 0xA9, 0x00,
	0x48, 0xF0, 0xC5, 0xA9, 0xB3, 0x48, 0xAD, 0x78,
	0x04, 0x18, 0x90, 0xEC, 0xA9, 0xBC, 0x48, 0xAD,
	0xF8, 0x04, 0x18, 0x90, 0xE3, 0xA9, 0x81, 0x48,
	0x7E, 0x38, 0x06, 0x90, 0x05, 0xAD, 0x78, 0x05,
	0xB0, 0xD6, 0x8A, 0x48, 0xA9, 0xD8, 0x48, 0xA9,
	0x0C, 0x9D, 0xB8, 0x05, 0xA9, 0x01, 0x48, 0xD0,
	0x97, 0xBD, 0x38, 0x06, 0x8D, 0x78, 0x05, 0x60,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC2,
	0xBD, 0x38, 0x07, 0x29, 0x0F, 0x09, 0x40, 0x9D,	// 0x0200
	0x38, 0x07, 0x8A, 0x48, 0x48, 0x48, 0xA9, 0x11,
	0xD0, 0x27, 0xA9, 0x1E, 0x48, 0xA9, 0x0C, 0x9D,
	0xB8, 0x05, 0xA9, 0x01, 0x48, 0xD0, 0x51, 0xAD,
	0xB3, 0xFB, 0xC9, 0x06, 0xD0, 0x21, 0xAD, 0x19,
	0xC0, 0x30, 0xFB, 0xAD, 0x19, 0xC0, 0x10, 0xFB,
	0xAD, 0x19, 0xC0, 0x30, 0xFB, 0xA9, 0x7F, 0xD0,
	0x00, 0x48, 0xA9, 0x50, 0x48, 0xA9, 0x0C, 0x9D,
	0xB8, 0x05, 0xA9, 0x00, 0x48, 0xF0, 0x29, 0xA5,
	0x06, 0x48, 0xA5, 0x07, 0x48, 0x98, 0x48, 0xA9,
	0x20, 0x85, 0x07, 0xA0, 0x00, 0x84, 0x06, 0xA9,
	0x00, 0x91, 0x06, 0xC8, 0xD0, 0xFB, 0xE6, 0x07,
	0xA5, 0x07, 0xC9, 0x40, 0xD0, 0xF1, 0x68, 0xA8,
	0xA5, 0x08, 0x48, 0xA9, 0x00, 0xF0, 0x1C, 0xFF,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0x30, 0x0A, 0xF0, 0x80,
	0xA9, 0x00, 0x9D, 0xB8, 0x05, 0x48, 0xF0, 0xE8,	// 0x0280
	0x60, 0xD0, 0xAE, 0xA9, 0x01, 0x8D, 0xD0, 0x3F,
	0x8D, 0xE0, 0x3F, 0xAD, 0x57, 0xC0, 0xAD, 0x54,
	0xC0, 0xAD, 0x52, 0xC0, 0xAD, 0x50, 0xC0, 0xEA,
	0x85, 0x06, 0x85, 0x07, 0x85, 0x08, 0xE6, 0x06,
	0xD0, 0x0E, 0xE6, 0x07, 0xD0, 0x0C, 0xE6, 0x08,
	0xA5, 0x08, 0xC9, 0x01, 0x90, 0x0A, 0xB0, 0x1F,
	0x08, 0x28, 0x08, 0x28, 0xA9, 0x00, 0xA5, 0x00,
	0xAD, 0xFF, 0xCF, 0xB9, 0x82, 0xC0, 0x4A, 0xEA,
	0xEA, 0xB0, 0xDB, 0xAD, 0xFF, 0xCF, 0xB9, 0x82,
	0xC0, 0x4A, 0xA5, 0x00, 0xEA, 0xB0, 0xCF, 0x68,
	0x85, 0x08, 0x68, 0x85, 0x07, 0x68, 0x85, 0x06,
	0xA9, 0xE3, 0xD0, 0xA5, 0xAD, 0x51, 0xC0, 0xAD,
	0x56, 0xC0, 0x18, 0x90, 0x93, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC1,
	0xBD, 0x38, 0x06, 0xC9, 0x20, 0xD0, 0x06, 0xA9,	// 0x0300
	0x7F, 0x69, 0x01, 0x70, 0x01, 0xB8, 0xB9, 0x82,
	0xC0, 0x30, 0xFB, 0xB9, 0x81, 0xC0, 0x29, 0xFB,
	0x99, 0x81, 0xC0, 0xA9, 0xFF, 0x99, 0x80, 0xC0,
	0xB9, 0x81, 0xC0, 0x09, 0x04, 0x99, 0x81, 0xC0,
	0xBD, 0x38, 0x06, 0x99, 0x80, 0xC0, 0xB9, 0x82,
	0xC0, 0x09, 0x20, 0x99, 0x82, 0xC0, 0xB9, 0x82,
	0xC0, 0x10, 0xFB, 0x29, 0xDF, 0x99, 0x82, 0xC0,
	0x70, 0x44, 0xBD, 0x38, 0x06, 0xC9, 0x30, 0xD0,
	0x35, 0xA9, 0x00, 0x9D, 0xB8, 0x04, 0x9D, 0xB8,
	0x03, 0x9D, 0x38, 0x05, 0x9D, 0x38, 0x04, 0xF0,
	0x25, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0xF0, 0x82, 0xA9, 0x00,
	0x9D, 0xB8, 0x05, 0x48, 0xF0, 0xEA, 0xB9, 0x81,	// 0x0380
	0xC0, 0x29, 0xFB, 0x99, 0x81, 0xC0, 0xA9, 0x00,
	0x99, 0x80, 0xC0, 0xB9, 0x81, 0xC0, 0x09, 0x04,
	0x99, 0x81, 0xC0, 0xB9, 0x82, 0xC0, 0x0A, 0x10,
	0xFA, 0xB9, 0x80, 0xC0, 0x9D, 0x38, 0x06, 0xB9,
	0x82, 0xC0, 0x09, 0x10, 0x99, 0x82, 0xC0, 0xB9,
	0x82, 0xC0, 0x0A, 0x30, 0xFA, 0xB9, 0x82, 0xC0,
	0x29, 0xEF, 0x99, 0x82, 0xC0, 0xBD, 0xB8, 0x06,
	0x29, 0xF1, 0x1D, 0x38, 0x06, 0x9D, 0xB8, 0x06,
	0x29, 0x0E, 0xD0, 0xB2, 0xA9, 0x00, 0x9D, 0xB8,
	0x05, 0xA9, 0x02, 0x48, 0xD0, 0x9A, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC3,
	0xE4, 0x37, 0xD0, 0x2D, 0xA9, 0x07, 0xC5, 0x36,	// 0x0400
	0xF0, 0x27, 0x85, 0x36, 0x68, 0xC9, 0x8D, 0xF0,
	0x74, 0x29, 0x01, 0x09, 0x80, 0x9D, 0x38, 0x07,
	0x8A, 0x48, 0xA9, 0x84, 0x48, 0xBD, 0x38, 0x07,
	0x4A, 0xA9, 0x80, 0xB0, 0x01, 0x0A, 0x48, 0xA9,
	0x0C, 0x9D, 0xB8, 0x05, 0xA9, 0x00, 0x48, 0xF0,
	0x3F, 0xE4, 0x39, 0xD0, 0xD7, 0xA9, 0x05, 0x85,
	0x38, 0xBD, 0x38, 0x07, 0x29, 0x01, 0xD0, 0x14,
	0x68, 0x68, 0x68, 0x68, 0xA9, 0x00, 0x9D, 0xB8,
	0x03, 0x9D, 0xB8, 0x04, 0x9D, 0x38, 0x04, 0x9D,
	0x38, 0x05, 0xF0, 0x3C, 0xBD, 0x38, 0x07, 0x29,
	0x01, 0x09, 0x80, 0x9D, 0x38, 0x07, 0x8A, 0x48,
	0xA9, 0xA1, 0x48, 0xA9, 0x10, 0x48, 0xA9, 0x0C,
	0xD0, 0x30, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0x30, 0x11, 0xF0, 0x80,
	0x6A, 0xB0, 0x89, 0x90, 0xB4, 0xA9, 0x00, 0x9D,	// 0x0480
	0xB8, 0x05, 0xA9, 0x01, 0x48, 0xD0, 0xE1, 0x60,
	0xA9, 0xC0, 0x9D, 0xB8, 0x06, 0x8C, 0x22, 0x02,
	0xA9, 0x0A, 0x9D, 0xB8, 0x05, 0xA9, 0x00, 0x48,
	0xF0, 0xCE, 0x68, 0x68, 0x68, 0x68, 0xA9, 0x05,
	0x9D, 0x38, 0x06, 0xB9, 0x81, 0xC0, 0x29, 0xFB,
	0x99, 0x81, 0xC0, 0xA9, 0x00, 0x99, 0x80, 0xC0,
	0xB9, 0x81, 0xC0, 0x09, 0x04, 0x99, 0x81, 0xC0,
	0xB9, 0x82, 0xC0, 0x0A, 0x10, 0xFA, 0xB9, 0x80,
	0xC0, 0x48, 0xB9, 0x82, 0xC0, 0x09, 0x10, 0x99,
	0x82, 0xC0, 0xB9, 0x82, 0xC0, 0x0A, 0x30, 0xFA,
	0xB9, 0x82, 0xC0, 0x29, 0xEF, 0x99, 0x82, 0xC0,
	0xDE, 0x38, 0x06, 0xD0, 0xDB, 0x68, 0x9D, 0xB8,
	0x06, 0x68, 0x9D, 0x38, 0x05, 0x68, 0x9D, 0x38,
	0x04, 0x68, 0x9D, 0xB8, 0x04, 0x68, 0x9D, 0xB8,
	0x03, 0x18, 0x90, 0x99, 0xFF, 0xFF, 0xFF, 0xC8,
	0x8A, 0x48, 0x48, 0x48, 0xA9, 0x12, 0x48, 0xBC,	// 0x0500
	0xB8, 0x03, 0xBD, 0xB8, 0x04, 0xAA, 0x98, 0xA0,
	0x05, 0xD0, 0x6D, 0xAE, 0xF8, 0x07, 0xA9, 0x24,
	0x48, 0xBC, 0x38, 0x04, 0xBD, 0x38, 0x05, 0xAA,
	0x98, 0xA0, 0x0C, 0xD0, 0x5B, 0xAE, 0xF8, 0x07,
	0xA9, 0x43, 0x48, 0xAD, 0x00, 0xC0, 0x0A, 0x08,
	0xBD, 0xB8, 0x06, 0x2A, 0x2A, 0x2A, 0x29, 0x03,
	0x49, 0x03, 0x38, 0x69, 0x00, 0x28, 0xA2, 0x00,
	0xA0, 0x10, 0xD0, 0x4D, 0xA9, 0x8D, 0x8D, 0x11,
	0x02, 0x48, 0xA9, 0x11, 0x48, 0x48, 0xA9, 0x00,
	0xF0, 0x12, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xAE, 0xF8, 0x07, 0xAC,
	0x22, 0x02, 0x9D, 0xB8, 0x05, 0xA9, 0x01, 0x48,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0x30, 0x4E, 0xF0, 0x80,
	0xE0, 0x80, 0x90, 0x0D, 0x49, 0xFF, 0x69, 0x00,	// 0x0580
	0x48, 0x8A, 0x49, 0xFF, 0x69, 0x00, 0xAA, 0x68,
	0x38, 0x8D, 0x21, 0x02, 0x8E, 0x20, 0x02, 0xA9,
	0xAB, 0x90, 0x02, 0xA9, 0xAD, 0x48, 0xA9, 0xAC,
	0x99, 0x01, 0x02, 0xA2, 0x11, 0xA9, 0x00, 0x18,
	0x2A, 0xC9, 0x0A, 0x90, 0x02, 0xE9, 0x0A, 0x2E,
	0x21, 0x02, 0x2E, 0x20, 0x02, 0xCA, 0xD0, 0xF0,
	0x09, 0xB0, 0x99, 0x00, 0x02, 0x88, 0xF0, 0x08,
	0xC0, 0x07, 0xF0, 0x04, 0xC0, 0x0E, 0xD0, 0xDB,
	0x68, 0x99, 0x00, 0x02, 0x60, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xCD,
	0xB8, 0x50, 0x13, 0xBD, 0x38, 0x07, 0x29, 0x01,	// 0x0600
	0xF0, 0x47, 0xA9, 0x10, 0x48, 0xA9, 0x05, 0x9D,
	0x38, 0x06, 0xA9, 0x7F, 0x69, 0x01, 0xB9, 0x82,
	0xC0, 0x30, 0xFB, 0xB9, 0x81, 0xC0, 0x29, 0xFB,
	0x99, 0x81, 0xC0, 0xA9, 0xFF, 0x99, 0x80, 0xC0,
	0xB9, 0x81, 0xC0, 0x09, 0x04, 0x99, 0x81, 0xC0,
	0x68, 0x99, 0x80, 0xC0, 0xB9, 0x82, 0xC0, 0x09,
	0x20, 0x99, 0x82, 0xC0, 0xB9, 0x82, 0xC0, 0x10,
	0xFB, 0x29, 0xDF, 0x99, 0x82, 0xC0, 0x70, 0x3F,
	0x70, 0x07, 0xBD, 0x38, 0x07, 0x4A, 0x4A, 0x4A,
	0x4A, 0xB8, 0x9D, 0xB8, 0x05, 0xF0, 0x02, 0xA9,
	0x80, 0x48, 0x50, 0x14, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0xF0, 0x82, 0xC9, 0x02,
	0xF0, 0x81, 0xD0, 0x02, 0xF0, 0xC2, 0xB8, 0xB9,	// 0x0680
	0x81, 0xC0, 0x29, 0xFB, 0x99, 0x81, 0xC0, 0xA9,
	0x00, 0x99, 0x80, 0xC0, 0xB9, 0x81, 0xC0, 0x09,
	0x04, 0x99, 0x81, 0xC0, 0xB9, 0x82, 0xC0, 0x0A,
	0x10, 0xFA, 0xB9, 0x80, 0xC0, 0x70, 0x05, 0x9D,
	0x38, 0x06, 0x50, 0x01, 0x48, 0xB9, 0x82, 0xC0,
	0x09, 0x10, 0x99, 0x82, 0xC0, 0xB9, 0x82, 0xC0,
	0x0A, 0x30, 0xFA, 0xB9, 0x82, 0xC0, 0x29, 0xEF,
	0x99, 0x82, 0xC0, 0x50, 0x19, 0xDE, 0x38, 0x06,
	0xD0, 0xD2, 0x68, 0x9D, 0xB8, 0x06, 0x68, 0x9D,
	0x38, 0x05, 0x68, 0x9D, 0x38, 0x04, 0x68, 0x9D,
	0xB8, 0x04, 0x68, 0x9D, 0xB8, 0x03, 0xA9, 0x00,
	0xF0, 0xA2, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xC1,
	0xBD, 0x38, 0x06, 0xC9, 0x40, 0xF0, 0x22, 0xC9,	// 0x0700
	0x60, 0xF0, 0x0D, 0xC9, 0x61, 0xF0, 0x09, 0xC9,
	0xA0, 0xD0, 0x2E, 0x48, 0xA9, 0x02, 0xD0, 0x45,
	0xAD, 0xF8, 0x05, 0x48, 0xAD, 0x78, 0x05, 0x48,
	0xAD, 0xF8, 0x04, 0x48, 0xAD, 0x78, 0x04, 0xB0,
	0x0F, 0xBD, 0x38, 0x05, 0x48, 0xBD, 0x38, 0x04,
	0x48, 0xBD, 0xB8, 0x04, 0x48, 0xBD, 0xB8, 0x03,
	0x48, 0xBD, 0x38, 0x06, 0x48, 0xA9, 0x05, 0xD0,
	0x1C, 0x29, 0x0C, 0x4A, 0x4A, 0x4A, 0xB0, 0x3E,
	0x4A, 0x90, 0x0C, 0xAD, 0x78, 0x05, 0x48, 0xBD,
	0x38, 0x06, 0x48, 0xA9, 0x02, 0xD0, 0x06, 0xBD,
	0x38, 0x06, 0x48, 0xA9, 0x01, 0x9D, 0x38, 0x06,
	0xD0, 0x4F, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xB9, 0x82, 0xC0, 0x29, 0xF1, 0x1D, 0xB8, 0x05,
	0x99, 0x82, 0xC0, 0x68, 0xD0, 0x82, 0xA9, 0x00,
	0x9D, 0xB8, 0x05, 0x48, 0xF0, 0xEA, 0x4A, 0xB0,	// 0x0780
	0x13, 0xAD, 0xF8, 0x04, 0x48, 0xAD, 0x78, 0x04,
	0x48, 0xBD, 0x38, 0x06, 0x48, 0xA9, 0x03, 0x9D,
	0x38, 0x06, 0xD0, 0x15, 0xAD, 0x78, 0x05, 0x48,
	0xAD, 0xF8, 0x04, 0x48, 0xAD, 0x78, 0x04, 0x48,
	0xBD, 0x38, 0x06, 0x48, 0xA9, 0x04, 0x9D, 0x38,
	0x06, 0xB9, 0x82, 0xC0, 0x30, 0xFB, 0xB9, 0x81,
	0xC0, 0x29, 0xFB, 0x99, 0x81, 0xC0, 0xA9, 0xFF,
	0x99, 0x80, 0xC0, 0xB9, 0x81, 0xC0, 0x09, 0x04,
	0x99, 0x81, 0xC0, 0x68, 0x99, 0x80, 0xC0, 0xB9,
	0x82, 0xC0, 0x09, 0x20, 0x99, 0x82, 0xC0, 0xB9,
	0x82, 0xC0, 0x10, 0xFB, 0x29, 0xDF, 0x99, 0x82,
	0xC0, 0xDE, 0x38, 0x06, 0xF0, 0x98, 0xB9, 0x82,
	0xC0, 0x30, 0xFB, 0x10, 0xD6, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xCE
};

WRITE_HANDLER( M6821_Listener_B )
{
	((CMouseCard*)objTo)->On6821_B( byData );
}

WRITE_HANDLER( M6821_Listener_A )
{
	((CMouseCard*)objTo)->On6821_A( byData );
}

CALLBACK_HANDLER( MouseHandler )
{
	((CMouseCard*)objTo)->OnMouseEvent();
}
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CMouseCard::CMouseCard()
{
	InitRomImage();
	m_iDeviceNum = CARD_MOUSE_INTERFACE;
	m_strDeviceName = "Mouse Interface";
	m_c6821.SetListenerB( this, M6821_Listener_B );
	m_c6821.SetListenerA( this, M6821_Listener_A );
	g_cDIMouse.SetMouseListener( this, MouseHandler );
	m_by6821A = 0;
	m_by6821B = 0x40;		// Set PB6
	m_c6821.SetPB(m_by6821B);
	m_bVBL = FALSE;
	Reset();
	memset( m_byBuff, 0, sizeof( m_byBuff ) );
}

CMouseCard::~CMouseCard()
{
	g_cDIMouse.SetMouseListener( NULL, NULL );
}

void CMouseCard::InitRomImage()
{
	m_pbyRom = (BYTE*)rom;
}

BYTE CMouseCard::Read(WORD wAddr)
{
	BYTE byRS;
	byRS = wAddr & 3;
	return m_c6821.Read( byRS );
}

void CMouseCard::Write(WORD wAddr, BYTE byData)
{
	BYTE byRS;
	byRS = wAddr & 3;
	m_c6821.Write( byRS, byData );
}

BYTE CMouseCard::ReadRom(WORD wAddr)
{
	return m_pbyRom[ ( ( m_by6821B << 7 ) & 0x0700 ) + ( wAddr & 0xFF ) ];
}

void CMouseCard::On6821_A(BYTE byData)
{
	m_by6821A = byData;
}

void CMouseCard::On6821_B(BYTE byData)
{
	BYTE byDiff = ( m_by6821B ^ byData ) & 0x3E;

	if ( byDiff )
	{
		m_by6821B &= ~0x3E;
		m_by6821B |= byData & 0x3E;
		if ( byDiff & BIT5 )			// Write to 0285 chip
		{
			if ( byData & BIT5 )
				m_by6821B |= BIT7;		// OK, I'm ready to read from MC6821
			else						// Clock Activate for read
			{
				m_byBuff[m_nBuffPos++] = m_by6821A;
				if ( m_nBuffPos == 1 )
					OnCommand();
				if ( m_nBuffPos == m_nDataLen || m_nBuffPos > 7 )
				{
					OnWrite();			// Have written all, Commit the command.
					m_nBuffPos = 0;
				}
				m_by6821B &= ~BIT7;		// for next reading
				m_c6821.SetPB( m_by6821B );
			}
			
		}
		if ( byDiff & BIT4 )		// Read from 0285 chip ?
		{
			if ( byData & BIT4 )
				m_by6821B &= ~BIT6;		// OK, I'll prepare next value
			else						// Clock Activate for write
			{
				if ( m_nBuffPos )		// if m_nBuffPos is 0, something goes wrong!
					m_nBuffPos++;
				if ( m_nBuffPos == m_nDataLen || m_nBuffPos > 7 )
					m_nBuffPos = 0;			// Have read all, ready for next command.
				else
					m_c6821.SetPA( m_byBuff[m_nBuffPos] );
				m_by6821B |= BIT6;		// for next writing
			}
		}
		m_c6821.SetPB( m_by6821B );
	}
}


void CMouseCard::OnCommand()
{
	switch( m_byBuff[0] & 0xF0 )
	{
	case MOUSE_SET:
		m_nDataLen = 1;
		m_byMode = m_byBuff[0] & 0x0F;
		break;
	case MOUSE_READ:				// Read
		m_nDataLen = 6;
		m_byState &= 0x20;
		m_nX = g_cDIMouse.GetX();
		m_nY = g_cDIMouse.GetY();
		if ( m_bBtn0 )	m_byState |= 0x40;			// Previous Button 0
		if ( m_bBtn1 )	m_byState |= 0x01;			// Previous Button 1
		m_bBtn0 = g_cDIMouse.IsButton0();
		m_bBtn1 = g_cDIMouse.IsButton1();
		if ( m_bBtn0 )	m_byState |= 0x80;			// Current Button 0
		if ( m_bBtn1 )	m_byState |= 0x10;			// Current Button 1
		m_byBuff[1] = m_nX & 0xFF;
		m_byBuff[2] = ( m_nX >> 8 ) & 0xFF;
		m_byBuff[3] = m_nY & 0xFF;
		m_byBuff[4] = ( m_nY >> 8 ) & 0xFF;
		m_byBuff[5] = m_byState;			// button 0/1 interrupt status
		m_byState &= ~0x20;
		break;
	case MOUSE_SERV:
		m_nDataLen = 2;
		m_byBuff[1] = m_byState & ~0x20;			// reason of interrupt
		break;
	case MOUSE_CLEAR:
		Reset();
		m_nDataLen = 1;
		break;
	case MOUSE_POS:
		m_nDataLen = 5;
		break;
	case MOUSE_INIT:
		m_nDataLen = 3;
		m_byBuff[1] = 0xFF;			// I don't know what it is
		break;
	case MOUSE_CLAMP:
		m_nDataLen = 5;
		break;
	case MOUSE_HOME:
		m_nDataLen = 1;
		g_cDIMouse.SetPosition( 0, 0, 0 );
		break;
	case MOUSE_TIME:		// 0x90
		switch( m_byBuff[0] & 0x0C )
		{
		case 0x00: m_nDataLen = 1; break;	// write cmd ( #$90 is DATATIME 60Hz, #$91 is 50Hz )
		case 0x04: m_nDataLen = 3; break;	// write cmd, $0478, $04F8
		case 0x08: m_nDataLen = 2; break;	// write cmd, $0578
		case 0x0C: m_nDataLen = 4; break;	// write cmd, $0478, $04F8, $0578
		}
		break;
	case 0xA0:
		m_nDataLen = 2;
		break;
	case 0xB0:
	case 0xC0:
		m_nDataLen = 1;
		break;
	default:
		m_nDataLen = 1;
		TRACE( "CMD : UNKNOWN CMD : #$%02X\n", m_byBuff[0] );
		break;
	}
	m_c6821.SetPA( m_byBuff[1] );
}

void CMouseCard::OnWrite()
{
	int nMin, nMax;
	switch( m_byBuff[0] & 0xF0 )
	{
	case MOUSE_CLAMP:
		nMin = ( m_byBuff[3] << 8 ) | m_byBuff[1];
		nMax = ( m_byBuff[4] << 8 ) | m_byBuff[2];
		if ( m_byBuff[0] & 1 )	// Clamp Y
			g_cDIMouse.ClampY( nMin, nMax );
		else					// Clamp X
			g_cDIMouse.ClampX( nMin, nMax );
		break;
	case MOUSE_POS:
		m_nX = ( m_byBuff[2] << 8 ) | m_byBuff[1];
		m_nY = ( m_byBuff[4] << 8 ) | m_byBuff[3];
		g_cDIMouse.SetPosition( m_nX, m_nY, 0 );
		break;
	case MOUSE_INIT:
		m_nX = 0;
		m_nY = 0;
		g_cDIMouse.ClampX( 0, 1023 );
		g_cDIMouse.ClampY( 0, 1023 );
		g_cDIMouse.SetPosition( 0, 0, 0 );
		break;
	}
}

void CMouseCard::OnMouseEvent()
{
	int byState = 0;
	int nX, nY;
	BOOL bBtn0, bBtn1;

	if ( !( m_byMode & 1 ) )		// Mouse Off
		return;

	nX = g_cDIMouse.GetX();
	nY = g_cDIMouse.GetY();
	bBtn0 = g_cDIMouse.IsButton0();
	bBtn1 = g_cDIMouse.IsButton1();
	if ( m_nX != nX || m_nY != nY )
		byState |= 0x22;				// X/Y moved since last READMOUSE | Movement interrupt
	if ( m_bBtn0 != bBtn0 || m_bBtn1 != bBtn1 )
		byState |= 0x04;				// Button 0/1 interrupt
	if ( m_bVBL )		byState |= 0x08;
	byState &= m_byMode & 0x2E;

	if ( byState & 0x0E )
	{
		m_byState |= byState;
		g_pBoard->m_cpu.Assert_IRQ();
	}
}

void CMouseCard::Clock(int nClock)
{
	BOOL bVBL = g_pBoard->IsVBlank();
	if ( m_bVBL != bVBL )
	{
		m_bVBL = bVBL;
		if ( m_bVBL )
			OnMouseEvent();
	}
}

void CMouseCard::Reset()
{
	m_nBuffPos = 0;
	m_nDataLen = 1;

	m_byMode = 0;
	m_byState = 0;
	m_nX = 0;
	m_nY = 0;
	m_bBtn0 = 0;
	m_bBtn1 = 0;
	g_cDIMouse.ClampX( 0, 1023 );
	g_cDIMouse.ClampY( 0, 1023 );
	g_cDIMouse.SetPosition( 0, 0, 0 );
}
