// SavDoc.cpp : implementation of the CSavDoc class
//
// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (c) Microsoft Corporation.  All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and related
// electronic documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

#include "stdafx.h"
#include "SavApp.h"

#include "SavDoc.h"
#include "SavView.h"
#include "SetVarDialog.h"
#include "SetSpecDlg.h"
#include "drawobj.h"
#include "summpage.h"
#include "statpage.h"
#include "model.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CSavDoc

IMPLEMENT_DYNCREATE(CSavDoc, COleDocument)

BEGIN_MESSAGE_MAP(CSavDoc, COleDocument)
	//{{AFX_MSG_MAP(CSavDoc)
	ON_COMMAND(ID_VIEW_PAPERCOLOR, OnViewPaperColor)
	ON_COMMAND(ID_FILE_SUMMARYINFO, OnFileSummaryInfo)
	//}}AFX_MSG_MAP
	// Enable default OLE container implementation
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE, COleDocument::OnUpdatePasteMenu)
	ON_UPDATE_COMMAND_UI(ID_EDIT_PASTE_LINK, COleDocument::OnUpdatePasteLinkMenu)
	ON_UPDATE_COMMAND_UI(ID_OLE_EDIT_LINKS, COleDocument::OnUpdateEditLinksMenu)
	ON_COMMAND(ID_OLE_EDIT_LINKS, COleDocument::OnEditLinks)
	ON_UPDATE_COMMAND_UI(ID_OLE_VERB_FIRST, COleDocument::OnUpdateObjectVerbMenu)
		// MAPI support
	ON_COMMAND(ID_FILE_SEND_MAIL, OnFileSendMail)
	ON_UPDATE_COMMAND_UI(ID_FILE_SEND_MAIL, OnUpdateFileSendMail)

	ON_COMMAND(ID_SET_VARIABLE, OnSetVariable)
	ON_COMMAND(ID_SET_SPEC, OnSetSpecification)
	ON_COMMAND(ID_VERIFY, OnVerify)

END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSavDoc construction/destruction

CSavDoc::CSavDoc()
{
	EnableCompoundFile();

	m_nMapMode = MM_ANISOTROPIC;
	m_paperColor = RGB(255, 255, 255);
	m_pSummInfo = NULL;
	ComputePageSize();
}

CSavDoc::~CSavDoc()
{
	POSITION pos = m_objects.GetHeadPosition();
	while (pos != NULL)
		delete m_objects.GetNext(pos);
	delete m_pSummInfo;
}

BOOL CSavDoc::OnNewDocument()
{
	if (!COleDocument::OnNewDocument())
		return FALSE;

	// reinitialization code
	// (SDI documents will reuse this document)
	if(m_pSummInfo != NULL)
		delete m_pSummInfo;
	m_pSummInfo = new CSummInfo;
	// Title, Subject, Author, Keywords default to empty string
	// Comments, Template, SavedBy default to empty string
	// LastSave, LastPrint, EditTime, RevNum default to 0
	m_pSummInfo->StartEditTimeCount();
	m_pSummInfo->RecordCreateDate();
	m_pSummInfo->SetNumPages(1);
	// NumWords, NumChars default to 0
	m_pSummInfo->SetAppname( _T("Situaltion-aware Application Verification tool") );
	// Security defaults to 0

	//
	// build new model
	//
	pModel = CModelBuilder::BuildModel();
	if (pModel)
	{
		//::AfxMessageBox("new model is built");
	}

	return TRUE;
}

BOOL CSavDoc::OnOpenDocument(LPCTSTR lpszPathName)
{
	if( m_pSummInfo != NULL)
		delete m_pSummInfo;
	m_pSummInfo = new CSummInfo;
	m_pSummInfo->StartEditTimeCount();
	return COleDocument::OnOpenDocument(lpszPathName);
}

BOOL CSavDoc::OnSaveDocument(LPCTSTR lpszPathName)
{
	m_pSummInfo->RecordSaveDate();
	m_pSummInfo->IncrRevNum();
	m_pSummInfo->SetLastAuthor(m_pSummInfo->GetAuthor());
	m_pSummInfo->AddCountToEditTime();
	m_pSummInfo->StartEditTimeCount();
	return COleDocument::OnSaveDocument(lpszPathName);
}

/////////////////////////////////////////////////////////////////////////////
// CSavDoc serialization

void CSavDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		ar << m_paperColor;
		m_objects.Serialize(ar);
		m_pSummInfo->WriteToStorage(m_lpRootStg);
	}
	else
	{
		ar >> m_paperColor;
		m_objects.Serialize(ar);
		m_pSummInfo->ReadFromStorage(m_lpRootStg);
	}

	// By calling the base class COleDocument, we enable serialization
	//  of the container document's COleClientItem objects automatically.
	COleDocument::Serialize(ar);
}


/////////////////////////////////////////////////////////////////////////////
// CSavDoc implementation

void CSavDoc::Draw(CDC* pDC, CSavView* pView)
{
	POSITION pos = m_objects.GetHeadPosition();
	while (pos != NULL)
	{
		CDrawObj* pObj = m_objects.GetNext(pos);
		pObj->Draw(pDC);
		if (pView->m_bActive && !pDC->IsPrinting() && pView->IsSelected(pObj))
			pObj->DrawTracker(pDC, CDrawObj::selected);
	}
}

void CSavDoc::Add(CDrawObj* pObj)
{
	m_objects.AddTail(pObj);

	pObj->m_pDocument = this;
	SetModifiedFlag();

	//
	// ⿡ 𵨿   Ͼ.
	//
	CDrawRect* rect = (CDrawRect*)pObj;
	if (rect->getShape() == CDrawRect::line) {
		if (rect->fromNode() && rect->toNode())
			pModel->createTransition(rect->fromNode(), rect->toNode());
	} else if (rect->getShape() == CDrawRect::ellipse) {
		CState* node = pModel->createState();
		if (node) {
			rect->SetNode(node);
		}
	}

	UpdateModelinfo();
}

void CSavDoc::Remove(CDrawObj* pObj)
{
	// Find and remove from document
	POSITION pos = m_objects.Find(pObj);
	if (pos != NULL)
		m_objects.RemoveAt(pos);
	// set document modified flag
	SetModifiedFlag();

	// call remove for each view so that the view can remove from m_selection
	pos = GetFirstViewPosition();
	while (pos != NULL)
		((CSavView*)GetNextView(pos))->Remove(pObj);

	//
	// ⿡ 𵨿   Ͼ.
	//
	CDrawRect* rect = (CDrawRect*)pObj;
	if (rect->getShape() == CDrawRect::line) {
		if (rect->fromNode() && rect->toNode())
			pModel->deleteTransition(rect->fromNode(), rect->toNode());
	} else if (rect->getShape() == CDrawRect::ellipse) {
		CState* node = rect->Node();
		pModel->deleteState(node);
		//
		//  ellipse  ġ line  ãƼ Ѵ.
		//
		vector<CDrawObj*> deletes;
		pos = m_objects.GetHeadPosition();
		while (pos) {
			CDrawRect* obj = (CDrawRect*)m_objects.GetNext(pos);
			if (obj && obj->getShape() == CDrawRect::line) {
				if (obj->LineFrom() == pObj || obj->LineTo() == pObj) {
					deletes.push_back(obj);
				}
			}
		}
		for (vector<CDrawObj*>::iterator itr=deletes.begin(); itr!=deletes.end(); itr++) {
			Remove(*itr);
		}
//		UpdateAllViews(NULL);
	}

	UpdateModelinfo();
}

/*
//
// ArrowLine  ǥ ٲ۴.
//
void CSavDoc::ModifyArrowLine(CDrawObj* pObj, CRect& rect)
{
	if (!pObj)
		return;

	CDrawRect* pRect = (CDrawRect*)pObj;
	if (pRect->getShape() == CDrawRect::line) {
		//pRect->m_position.mo
		pRect->m_position = rect;
	}
}
*/

// point is in logical coordinates
CDrawObj* CSavDoc::ObjectAt(const CPoint& point)
{
	CRect rect(point, CSize(1, 1));
	POSITION pos = m_objects.GetTailPosition();
	while (pos != NULL)
	{
		CDrawObj* pObj = m_objects.GetPrev(pos);
		if (pObj->Intersects(rect))
			return pObj;
	}

	return NULL;
}

CDrawObj* CSavDoc::ObjectAt(const CPoint& point, DrawShape nDrawShape)
{
	CRect rect(point, CSize(1, 1));
	POSITION pos = m_objects.GetTailPosition();
	while (pos != NULL)
	{
		CDrawObj* pObj = m_objects.GetPrev(pos);
		if (pObj->Intersects(rect)) {
			if (pObj->IsKindOf(RUNTIME_CLASS(CDrawRect)) && 
				((CDrawRect*)pObj)->getShape() == nDrawShape) {
				return pObj;
			}
		}
	}

	return NULL;
}

void CSavDoc::ComputePageSize()
{
	CSize new_size(850, 1100);  // 8.5" x 11" default

	CPrintDialog dlg(FALSE);
	if (AfxGetApp()->GetPrinterDeviceDefaults(&dlg.m_pd))
	{
		// GetPrinterDC returns a HDC so attach it
		CDC dc;
		HDC hDC= dlg.CreatePrinterDC();
		ASSERT(hDC != NULL);
		dc.Attach(hDC);

		// Get the size of the page in loenglish
		new_size.cx = MulDiv(dc.GetDeviceCaps(HORZSIZE), 1000, 254);
		new_size.cy = MulDiv(dc.GetDeviceCaps(VERTSIZE), 1000, 254);
	}

	// if size changed then iterate over views and reset
	if (new_size != m_size)
	{
		m_size = new_size;
		POSITION pos = GetFirstViewPosition();
		while (pos != NULL)
			((CSavView*)GetNextView(pos))->SetPageSize(m_size);
	}
}

void CSavDoc::OnViewPaperColor()
{
	CColorDialog dlg;
	if (dlg.DoModal() != IDOK)
		return;

	m_paperColor = dlg.GetColor();
	SetModifiedFlag();
	UpdateAllViews(NULL);
}

/////////////////////////////////////////////////////////////////////////////
// CSavDoc diagnostics

#ifdef _DEBUG
void CSavDoc::AssertValid() const
{
	COleDocument::AssertValid();
}

void CSavDoc::Dump(CDumpContext& dc) const
{
	COleDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CSavDoc commands

void CSavDoc::OnSetSpecification()
{
	CSetSpecDlg dlg(pModel);
	if (dlg.DoModal() == IDOK)
		UpdateModelinfo();

}

void CSavDoc::OnVerify()
{
	vector<CState*> states;
	pModel->find_satisfies_Spec(CString("Open"), states);
	UpdateModelinfo();
}

void CSavDoc::OnSetVariable()
{
	CSetVarDialog svd(pModel);
	if (svd.DoModal() == IDOK)
		UpdateModelinfo();
}

void CSavDoc::OnFileSummaryInfo()
{
	ASSERT_VALID(this);

	CPropertySheet sheet( _T("Document Properties") );
	CSummPage summ;
	CStatPage stat;
	sheet.AddPage( &summ );
	sheet.AddPage( &stat );

	summ.m_strAppname = m_pSummInfo->GetAppname();
	summ.m_strTitle   = m_pSummInfo->GetTitle();
	summ.m_strSubj    = m_pSummInfo->GetSubject();
	summ.m_strAuthor  = m_pSummInfo->GetAuthor();
	summ.m_strKeywd   = m_pSummInfo->GetKeywords();
	summ.m_strCmt     = m_pSummInfo->GetComments();
	summ.m_strTempl   = m_pSummInfo->GetTemplate();

	stat.m_strSavedBy    = m_pSummInfo->GetLastAuthor();
	stat.m_strRevNum     = m_pSummInfo->GetRevNum();
	stat.m_strEditTime   = m_pSummInfo->GetEditTime();
	stat.m_strLastPrint  = m_pSummInfo->GetLastPrintDate();
	stat.m_strCreateDate = m_pSummInfo->GetCreateDate();
	stat.m_strLastSave   = m_pSummInfo->GetLastSaveDate();
	stat.m_strNumPages   = m_pSummInfo->GetNumPages();
	stat.m_strNumWords   = m_pSummInfo->GetNumWords();
	stat.m_strNumChars   = m_pSummInfo->GetNumChars();
	stat.m_strSecurity   = m_pSummInfo->GetSecurity();

	if (sheet.DoModal() != IDOK)
		return;

	m_pSummInfo->SetAuthor(summ.m_strAuthor);
	m_pSummInfo->SetKeywords(summ.m_strKeywd);
	m_pSummInfo->SetSubject(summ.m_strSubj);
	m_pSummInfo->SetComments(summ.m_strCmt);
	m_pSummInfo->SetTemplate(summ.m_strTempl);
	m_pSummInfo->SetTitle(summ.m_strTitle);

	SetModifiedFlag();
}

void CSavDoc::UpdateModelinfo()
{
	CString _text;
	pModel->trace(_text);

	POSITION pos = m_objects.GetTailPosition();
	while (pos != NULL)
	{
		CDrawRect* pObj = (CDrawRect*)m_objects.GetPrev(pos);
		if (pObj->getShape() == CDrawRect::text)
		{
			pObj->setText(_text);
		}
	}

	pModel->trace();

	UpdateAllViews(NULL);
}
